/*
   Determine how flushing an output stream effects performance time.

   This program writes 6,000,000 bytes through a pipe to its child
   process. There are two parameters that you can play with: the
   length of the lines that the bytes are "organized" into, and
   whether or not to call flush() whenever an output line is sent
   to the child.

   Experiment with these two parameters to see how they effect
   performance time.
*/
import java.util.Scanner;
import java.io.*;

public class Ex5_TestStreamFlush
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Command line for running child process.
      String[] cmd = {"java", "Ex5_TestStreamFlush_Child"};
      // Execute the command line.
      Process process = Runtime.getRuntime().exec(cmd);

      // A stream that writes data to child's stdin stream.
      PrintStream stdinOfChild =  new PrintStream( process.getOutputStream() );

      // A stream that reads data from the child's stdout stream.
      final Scanner stdoutOfChild = new Scanner( process.getInputStream() );

      // Create a "pump" between the child's output stream and stdout.
      new Thread(new Runnable(){public void run()
      {while(stdoutOfChild.hasNextLine())
       {String oneLine = stdoutOfChild.nextLine(); // read from output of child
        System.out.println( oneLine );             // write to standard out
       }}}).start();


      final int LENGTH_OF_LINE = 10;

      long startTime = System.currentTimeMillis(); // "start" the timer

      // Write 6,000,000 bytes to child's input buffer.
      for (int counter = 0; counter < 6000000; counter++)
      {
         stdinOfChild.print("a");
         if( (counter % LENGTH_OF_LINE) == 0 ) // How does this effect performance?
         {  counter++;
            stdinOfChild.println();
            //stdinOfChild.flush();   // How does this effect performance?
         }
      }
      stdinOfChild.close();

      process.waitFor();

      long endTime = System.currentTimeMillis();  // "stop" the "timer

      System.out.println("Total execution time (wct): " + (endTime - startTime) );
   }
}

/*
   When the length of the lines is very small (< 10) it seriously
   hurts performance to flush() the stdinOfChild stream after every
   line. That is because of the high number of calls to flush() and
   the fact that each flush() moves only two bytes. But when the
   length of a line is large (> 100), the calls to flush() no longer
   are important. There aren't as many of them and each call to flush
   moves a reasonable amount of data. In this case, calling flush() is
   no beeter or worse than relying on the internal buffers of the streams.

   However, notice that the best performance is with a large line
   length, 10,000, and no flushing.

   Curiously, when the line length because very large, 100,000 or
   1,000,000, the performance get very bad again.
*/